<?php
require_once __DIR__ . '/includes/auth.php';
require_login();

/**
 * current_user() is expected to return at least:
 *   ['id' => int, 'name' => string, 'role_id' => int?, 'role_name' => string?]
 * $con is assumed to be available via your bootstrap in auth.php (includes/initialize.php)
 */
$u = current_user();

/* ============================================================
   Tiny helpers
   ============================================================ */
function table_exists(mysqli $con, string $table): bool {
  $t = mysqli_real_escape_string($con, $table);
  $r = mysqli_query($con, "SHOW TABLES LIKE '$t'");
  return ($r && mysqli_num_rows($r) > 0);
}
function col_exists(mysqli $con, string $table, string $col): bool {
  $t = mysqli_real_escape_string($con, $table);
  $c = mysqli_real_escape_string($con, $col);
  $r = mysqli_query($con, "SHOW COLUMNS FROM `$t` LIKE '$c'");
  return ($r && mysqli_num_rows($r) > 0);
}

/* ============================================================
   Role resolver (only if current_user() lacks role_id)
   - Uses jos_admin_users_roles (your table name)
   - Picks first active role by orderby then id
   ============================================================ */
function pacific_resolve_role(mysqli $con, int $user_id): ?array {
  if ($user_id <= 0) return null;

  // Ensure dependent tables exist
  if (!table_exists($con, 'jos_admin_users_roles') || !table_exists($con, 'jos_admin_roles')) {
    return null;
  }
  if (!col_exists($con, 'jos_admin_users_roles', 'user_id') || !col_exists($con, 'jos_admin_users_roles', 'role_id')) {
    return null;
  }
  if (!col_exists($con, 'jos_admin_roles', 'id') || !col_exists($con, 'jos_admin_roles', 'name')) {
    return null;
  }
  // Optional columns checks (status, orderby) are lenient
  $has_status  = col_exists($con, 'jos_admin_roles', 'status');
  $has_orderby = col_exists($con, 'jos_admin_roles', 'orderby');

  $sql = "
    SELECT r.id AS role_id, r.name AS role_name
    FROM jos_admin_users_roles ur
    JOIN jos_admin_roles r ON r.id = ur.role_id
    WHERE ur.user_id = ?
  ";
  if ($has_status)  $sql .= " AND r.status = 1";
  if ($has_orderby) $sql .= " ORDER BY r.orderby ASC, r.id ASC";
  else              $sql .= " ORDER BY r.id ASC";
  $sql .= " LIMIT 1";

  $st = $con->prepare($sql);
  $st->bind_param('i', $user_id);
  $st->execute();
  $res = $st->get_result();
  $row = $res->fetch_assoc();
  $st->close();

  return $row ?: null;
}

/* ============================================================
   STRICT ACL menu fetcher
   - Requires jos_admin_rolemenus + jos_admin_menus
   - No admin override, no fallback.
   ============================================================ */
if (!function_exists('pacific_fetch_menu_sections')) {
  function pacific_fetch_menu_sections(mysqli $con, array $user): array {

    // Resolve role_id strictly
    $role_id = (int)($user['role_id'] ?? 0);
    if ($role_id <= 0) {
      // Try to resolve from jos_admin_users_roles once
      $maybe = pacific_resolve_role($con, (int)($user['id'] ?? 0));
      if (!empty($maybe['role_id'])) {
        $role_id = (int)$maybe['role_id'];
      }
    }
    if ($role_id <= 0) {
      // Unknown role -> no menus
      return [];
    }

    // Enforce ACL presence/shape
    $acl_ok = table_exists($con, 'jos_admin_rolemenus')
           && col_exists($con, 'jos_admin_rolemenus', 'menu_id')
           && col_exists($con, 'jos_admin_rolemenus', 'role_id')
           && col_exists($con, 'jos_admin_rolemenus', 'can_view');

    if (!$acl_ok) {
      return [];
    }

    // Ensure menus table/columns exist
    if (!table_exists($con, 'jos_admin_menus')) return [];
    $need_cols = ['id','menu_name','menu_link','icon','parent_id','orderby','status'];
    foreach ($need_cols as $c) {
      if (!col_exists($con, 'jos_admin_menus', $c)) return [];
    }
    $has_section = col_exists($con, 'jos_admin_menus', 'section');

    $sql = "
      SELECT 
        m.id, m.menu_name, m.menu_link, m.icon, m.parent_id, m.orderby,
        " . ($has_section
              ? "COALESCE(m.section, CASE WHEN m.parent_id = 0 THEN 'Main' ELSE 'Sub' END)"
              : "CASE WHEN m.parent_id = 0 THEN 'Main' ELSE 'Sub' END"
            ) . " AS section
      FROM jos_admin_menus m
      INNER JOIN jos_admin_rolemenus rm
              ON rm.menu_id = m.id
             AND rm.can_view = 1
      WHERE m.status = 1
        AND rm.role_id = ?
      ORDER BY m.parent_id ASC, m.orderby ASC, m.menu_name ASC
    ";
    $st = $con->prepare($sql);
    $st->bind_param('i', $role_id);
    $st->execute();
    $res  = $st->get_result();
    $rows = [];
    while ($r = $res->fetch_assoc()) { $rows[] = $r; }
    $st->close();

    // Build parent/child tree
    $byId = [];
    foreach ($rows as $r) { $r['children'] = []; $byId[$r['id']] = $r; }
    $roots = [];
    foreach ($byId as $id => &$node) {
      if (!empty($node['parent_id']) && isset($byId[$node['parent_id']])) {
        $byId[$node['parent_id']]['children'][] = &$node;
      } else {
        $roots[] = &$node;
      }
    }
    unset($node);

    // Group by section
    $sections = [];
    foreach ($roots as $r) {
      $sec = $r['section'] ?: 'Main';
      $sections[$sec][] = $r;
    }
    return $sections;
  }
}

/* ============================================================
   Example: fetch sections for your sidebar renderer (if any)
   ============================================================ */
$sections = [];
if (isset($con) && $con instanceof mysqli) {
  $sections = pacific_fetch_menu_sections($con, $u);
}

/* ============================================================
   Standalone render (no external shell include)
   ============================================================ */
$GLOBALS['PACIFIC_PAGE_TITLE'] = 'Dashboard';

ob_start(); ?>
<link rel="stylesheet" href="/adminconsole/assets/ui.css">

<div class="master-wrap">
  <div class="headbar">
    <div class="headbar-left"><h2 style="margin:0">Dashboard</h2></div>
    <div class="headbar-right"></div>
  </div>

  <div class="card" style="max-width:920px">
    <h3 style="margin-top:0">Welcome, <?= htmlspecialchars($u['name'] ?? 'User') ?> 👋</h3>
    <p style="color:#9ca3af;margin-top:6px">
      This console uses a <strong>strict ACL</strong> for the sidebar:
      <code>jos_admin_menus</code> × <code>jos_admin_rolemenus</code>.
      No admin override, no fallback.
    </p>



  </div>
</div>

<?php echo ob_get_clean();
