<?php
// Promoter User Master with full profile_type=3 linkage + free plan + cascading delete
// Single-card horizontal form; "City" label
// Path: /adminconsole/masters/create_promoter.php

require_once __DIR__ . '/../includes/auth.php';
require_login();
require_once __DIR__ . '/../includes/initialize.php'; // $con, csrf_token(), verify_csrf()

/* -------- page config -------- */
$default_page_title  = 'Create Promoter User';
$TABLE       = 'jos_app_promoter_profile';
$USER_TABLE  = 'jos_app_users';
$SUB_LOG     = 'jos_app_usersubscriptionlog';
$PLAN_TABLE  = 'jos_app_subscription_plans';
$PROMOTER_DEFAULT_PLAN_ID = 10; // as per API
ob_start();

/* -------- helpers -------- */
function h($s){ return htmlspecialchars((string)$s, ENT_QUOTES, 'UTF-8'); }
function clean($v){ return trim((string)$v); }

/**
 * keep_params
 * - $changes: assoc array of changes (set to null to unset)
 * - $include_script: when true returns full path like "/adminconsole/masters/create_promoter.php?x=1"
 *                    when false returns only the query string like "?x=1" or empty string
 */
function keep_params(array $changes=[],$include_script=true){
  $qs = $_GET;
  foreach($changes as $k=>$v){
    if ($v === null){
      unset($qs[$k]);
    } else {
      $qs[$k] = $v;
    }
  }
  $q = http_build_query($qs);
  $qs_str = $q ? ('?'.$q) : '';
  if ($include_script){
    // Use PHP_SELF (path relative to document root) to ensure links always point to this script.
    $script = $_SERVER['PHP_SELF'] ?? '';
    return $script . $qs_str;
  }
  return $qs_str;
}

function ensure_schema(mysqli $con, $table){
  $r = mysqli_query($con,"SHOW COLUMNS FROM `$table` LIKE 'status'");
  if(!$r || mysqli_num_rows($r)==0){ mysqli_query($con,"ALTER TABLE `$table` ADD `status` TINYINT(1) NOT NULL DEFAULT 1"); }
  $r = mysqli_query($con,"SHOW COLUMNS FROM `$table` LIKE 'orderby'");
  if(!$r || mysqli_num_rows($r)==0){ mysqli_query($con,"ALTER TABLE `$table` ADD `orderby` INT NOT NULL DEFAULT 0"); }
}
function back_to_list($msg='Saved'){
  $loc = keep_params(['add'=>null,'edit'=>null,'ok'=>$msg]);
  header("Location: ".$loc); exit;
}
/**
 * redirect with improved "return" support:
 * - if ?return=... is present, redirect there and append ok=...
 * - otherwise behave like back_to_list
 */
function redirect_after_action($msg='Saved'){
  if(!empty($_GET['return'])){
    // preserve existing query parameters on return if needed
    $return = (string)$_GET['return'];
    $sep = (strpos($return,'?')===false) ? '?' : '&';
    header("Location: " . $return . $sep . 'ok=' . rawurlencode($msg));
    exit;
  }
  back_to_list($msg);
}

function generateRandomCode($length = 4){
  $chars = 'ABCDEFGHIJKLMNOPQRSTUVWXYZ0123456789';
  return substr(str_shuffle(str_repeat($chars, $length)), 0, $length);
}
function build_my_referral_code($user_id){
  $secretKey = "MySecretKey123";
  $hashPart  = strtoupper(substr(md5($secretKey . $user_id), 0, 8));
  $randomPart = generateRandomCode(4);
  return "REFPAC-$hashPart-$randomPart";
}

/* -------- user access control (lightweight, non-destructive) --------
   How it works:
   - If $_SESSION['admin_permissions'] exists and is an array of strings,
     it checks for permissions named:
       - promoter.view
       - promoter.add
       - promoter.edit
       - promoter.delete
   - If that session key is absent, it allows all actions (maintains compatibility).
   - Server-side enforcement is applied on POST actions.
*/
function _session_permissions(){
  if(!empty($_SESSION['admin_permissions']) && is_array($_SESSION['admin_permissions'])){
    return $_SESSION['admin_permissions'];
  }
  return null;
}
function user_can($cap){
  // cap examples: 'promoter.view','promoter.add', 'promoter.edit','promoter.delete'
  $perms = _session_permissions();
  if($perms === null){
    // no explicit permissions system in session -> allow (backwards compatible)
    return true;
  }
  return in_array($cap, $perms, true);
}

/* -------- robust title loader for jos_admin_menus --------
   Tries a list of candidate columns (title, name, menu_title, label, text, caption).
   Falls back to provided default if no suitable column exists.
*/
function load_title_from_menu(mysqli $con, $default){
  $path = $_SERVER['PHP_SELF'] ?? '';
  $basename = basename($path);
  $title = $default;

  // get column names for the table
  $cols = [];
  $q = $con->prepare("
    SELECT COLUMN_NAME
    FROM INFORMATION_SCHEMA.COLUMNS
    WHERE TABLE_SCHEMA = DATABASE()
      AND TABLE_NAME = ?
  ");
  if($q){
    $tbl = 'jos_admin_menus';
    $q->bind_param('s', $tbl);
    $q->execute();
    $res = $q->get_result();
    while($res && ($c = $res->fetch_assoc())){
      $cols[] = $c['COLUMN_NAME'];
    }
  }

  // candidate names in preferred order
  $candidates = ['title','name','menu_title','label','text','caption'];

  $chosenCol = null;
  foreach($candidates as $cand){
    if(in_array($cand, $cols, true)){
      $chosenCol = $cand;
      break;
    }
  }

  if(!$chosenCol){
    // If we couldn't find any candidate column, return default
    return $title;
  }

  // safe quoting for column name (we already validated it exists)
  $col = $chosenCol;

  // Try an exact match by path/link
  $st = $con->prepare("SELECT `$col` AS found_title FROM `jos_admin_menus` WHERE path = ? OR link = ? LIMIT 1");
  if($st){
    $st->bind_param("ss", $path, $path);
    $st->execute();
    $res = $st->get_result();
    if($res && ($row = $res->fetch_assoc()) && !empty($row['found_title'])){
      return $row['found_title'];
    }
  }

  // Fallback: try basename match
  $st2 = $con->prepare("SELECT `$col` AS found_title FROM `jos_admin_menus` WHERE path LIKE CONCAT('%',?,'%') OR link LIKE CONCAT('%',?,'%') LIMIT 1");
  if($st2){
    $st2->bind_param("ss", $basename, $basename);
    $st2->execute();
    $res2 = $st2->get_result();
    if($res2 && ($r2 = $res2->fetch_assoc()) && !empty($r2['found_title'])){
      return $r2['found_title'];
    }
  }

  return $title;
}

ensure_schema($con,$TABLE);
$page_title = load_title_from_menu($con, $default_page_title);

/* -------- modes -------- */
$mode_add  = isset($_GET['add']);
$mode_edit = isset($_GET['edit']);
$edit_id   = $mode_edit ? (int)$_GET['edit'] : 0;

/* current actor capabilities (compute once) */
$can_view   = user_can('promoter.view');
$can_add    = user_can('promoter.add');
$can_edit   = user_can('promoter.edit');
$can_delete = user_can('promoter.delete');

/* -------- POST -------- */
$err='';
if ($_SERVER['REQUEST_METHOD']==='POST'){
  verify_csrf($_POST['csrf_token'] ?? ''); // pass token required by initialize.php
  $action = $_POST['_action'] ?? '';

  if ($action==='save'){
    $id        = (int)($_POST['id'] ?? 0);

    // permission enforcement
    if ($id>0 && !$can_edit){ $err='Permission denied: cannot edit.'; }
    if ($id===0 && !$can_add){ $err='Permission denied: cannot add.'; }

    $name      = clean($_POST['name'] ?? '');
    $mobile    = clean($_POST['mobile_no'] ?? '');
    $pan_no    = clean($_POST['pan_no'] ?? '');
    $city_id   = clean($_POST['city'] ?? '');      // label: City (STRING)
    $address   = clean($_POST['address'] ?? '');
    $status    = (int)($_POST['status'] ?? 1);

    if ($err === '' && ($name==='' || $mobile==='')){ $err='Name and Mobile are required.'; }

    if ($err===''){
      $con->begin_transaction();
      try {
        if ($id>0){
          // UPDATE promoter
          $sql="UPDATE `$TABLE` SET name=?, mobile_no=?, pan_no=?, city_id=?, address=?, status=? WHERE id=?";
          $st=$con->prepare($sql);
          if(!$st) throw new Exception('Prepare failed (update promoter): '.$con->error);
          $st->bind_param("ssssiii",$name,$mobile,$pan_no,$city_id,$address,$status,$id);
          $st->execute();
          if($st->errno){ throw new Exception('Update failed: '.$st->error); }

          // Sync user if linked
          $usrq = $con->prepare("SELECT userid FROM `$TABLE` WHERE id=?");
          $usrq->bind_param("i",$id); $usrq->execute();
          $usr = $usrq->get_result()->fetch_assoc();
          $linked_uid = (int)($usr['userid'] ?? 0);
          if ($linked_uid>0){
            $uu = $con->prepare("UPDATE `$USER_TABLE` SET mobile_no=?, city_id=?, address=?, status_id=1 WHERE id=?");
            $uu->bind_param("sssi",$mobile,$city_id,$address,$linked_uid);
            $uu->execute();
            if ($uu->errno){ throw new Exception('User sync failed: '.$uu->error); }
          }

          $con->commit();
          redirect_after_action('Updated');

        } else {
          // INSERT promoter (lat/long NULL; photo empty; orderby auto)
          $rb = mysqli_query($con,"SELECT IFNULL(MAX(orderby),0)+1 AS nexto FROM `$TABLE`");
          $orderby = ($rb && ($row=mysqli_fetch_assoc($rb))) ? (int)$row['nexto'] : 1;

          $sql="INSERT INTO `$TABLE` (name,mobile_no,pan_no,city_id,address,profile_photo,latitude,longitude,status,orderby)
                VALUES (?,?,?,?,?,'',NULL,NULL,?,?)";
          $st=$con->prepare($sql);
          if(!$st) throw new Exception('Prepare failed (insert promoter): '.$con->error);
          $st->bind_param("sssssii",$name,$mobile,$pan_no,$city_id,$address,$status,$orderby);
          $st->execute();
          if($st->errno){ throw new Exception('Insert failed: '.$st->error); }
          $profile_id = $st->insert_id;

          // USER: if exists (same mobile) -> link/update; else insert
          $chk = $con->prepare("SELECT id, profile_type_id, profile_id, myreferral_code FROM `$USER_TABLE` WHERE mobile_no=? LIMIT 1");
          $chk->bind_param("s",$mobile);
          $chk->execute();
          $existing = $chk->get_result()->fetch_assoc();

          if ($existing){
            $user_id = (int)$existing['id'];

            // ensure it points to promoter profile
            $uu = $con->prepare("UPDATE `$USER_TABLE` 
                SET profile_type_id=3, profile_id=?, city_id=?, address=?, status_id=1 
                WHERE id=?");
            $uu->bind_param("isssi",$profile_id,$city_id,$address,$user_id);
            $uu->execute();
            if ($uu->errno){ throw new Exception('User relink failed: '.$uu->error); }

            // ensure myreferral_code exists
            if (empty($existing['myreferral_code'])){
              $myref = build_my_referral_code($user_id);
              $updref = $con->prepare("UPDATE `$USER_TABLE` SET myreferral_code=? WHERE id=?");
              $updref->bind_param("si",$myref,$user_id);
              $updref->execute();
            }
          } else {
            // fresh user
            $hashed_password = password_hash($mobile, PASSWORD_BCRYPT);
            $profile_type_id = 3;
            $referred_by     = 0;
            $referral_code   = '';
            $fcm_token       = '';
            $latitude_null   = null;
            $longitude_null  = null;
            $active_plan_id  = $PROMOTER_DEFAULT_PLAN_ID;

            $stmt_user = $con->prepare("
              INSERT INTO `$USER_TABLE`
                (mobile_no, password, profile_type_id, profile_id, city_id, address, latitude, longitude,
                 referral_code, referred_by, active_plan_id, fcm_token, status_id, created_at)
              VALUES (?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, ?, 1, NOW())
            ");
            if(!$stmt_user) throw new Exception('Prepare failed (insert user): '.$con->error);
            // note: bind_param requires variables; nulls need to be passed as null values
            $lat_null = null;
            $lng_null = null;
            $stmt_user->bind_param(
              "ssiisssssiis",
              $mobile,
              $hashed_password,
              $profile_type_id,
              $profile_id,
              $city_id,
              $address,
              $lat_null,
              $lng_null,
              $referral_code,
              $referred_by,
              $active_plan_id,
              $fcm_token
            );
            $stmt_user->execute();
            if ($stmt_user->errno){ throw new Exception('User insert failed: '.$stmt_user->error); }
            $user_id = $stmt_user->insert_id;

            // myreferral_code
            $myref = build_my_referral_code($user_id);
            $updref = $con->prepare("UPDATE `$USER_TABLE` SET myreferral_code=? WHERE id=?");
            $updref->bind_param("si",$myref,$user_id);
            $updref->execute();
          }

          // Backfill userid into promoter
          $up = $con->prepare("UPDATE `$TABLE` SET userid=? WHERE id=?");
          $up->bind_param("ii",$user_id,$profile_id);
          $up->execute();

          // Free subscription (insert only if not already present for this user/profile/plan)
          $planq = $con->prepare("SELECT validity_months FROM `$PLAN_TABLE` WHERE id = ?");
          $planq->bind_param("i",$PROMOTER_DEFAULT_PLAN_ID);
          $planq->execute(); $planres=$planq->get_result();
          if ($planres && $planres->num_rows>0){
            $existsSub = $con->prepare("SELECT 1 FROM `$SUB_LOG` WHERE userid=? AND plan_id=? AND profile_type_id=3 AND profile_id=? AND payment_id='free_signup' LIMIT 1");
            $existsSub->bind_param("iii",$user_id,$PROMOTER_DEFAULT_PLAN_ID,$profile_id);
            $existsSub->execute(); $es = $existsSub->get_result()->fetch_assoc();

            if (!$es){
              $plan_row = $planres->fetch_assoc();
              $validity_months = (int)$plan_row['validity_months'];
              $current_year = date('Y'); $invoice_prefix = 'APP'.$current_year.'-';
              $res = mysqli_query($con,"SELECT MAX(CAST(SUBSTRING(invoiceno, LENGTH('$invoice_prefix')+1) AS UNSIGNED)) AS max_invoice FROM `$SUB_LOG` WHERE invoiceno LIKE '$invoice_prefix%'");
              $last_invoice = ($res && ($row=mysqli_fetch_assoc($res))) ? (int)$row['max_invoice'] : 0;
              $new_invoiceno = $invoice_prefix . str_pad($last_invoice+1,5,'0',STR_PAD_LEFT);

              $sub = $con->prepare("INSERT INTO `$SUB_LOG`
                (userid,plan_id,start_date,end_date,payment_status,amount_paid,payment_id,invoiceno,profile_type_id,profile_id)
                VALUES (?, ?, NOW(), DATE_ADD(NOW(), INTERVAL ? MONTH),'success',0,'free_signup',?,3,?)");
              $sub->bind_param("iiisi",$user_id,$PROMOTER_DEFAULT_PLAN_ID,$validity_months,$new_invoiceno,$profile_id);
              $sub->execute();
            }
          }

          $con->commit();
          redirect_after_action('Saved');
        }
      } catch (Exception $e){
        $con->rollback();
        $err=$e->getMessage();
      }
    }
  }

  elseif ($action==='delete'){
    $id=(int)($_POST['id'] ?? 0);
    if ($id>0){
      if (!$can_delete){ $err='Permission denied: cannot delete.'; }
      if ($err===''){
        $con->begin_transaction();
        try {
          // get linked user id (if any)
          $g = $con->prepare("SELECT userid FROM `$TABLE` WHERE id=? LIMIT 1");
          $g->bind_param("i",$id); $g->execute();
          $gr=$g->get_result(); $urow=$gr?$gr->fetch_assoc():null;
          $linked_uid = (int)($urow['userid'] ?? 0);

          if ($linked_uid>0){
            // delete subscription logs for THIS profile/user
            $delSub = $con->prepare("DELETE FROM `$SUB_LOG` WHERE userid=? AND profile_type_id=3 AND profile_id=?");
            $delSub->bind_param("ii",$linked_uid,$id);
            $delSub->execute();
            if ($delSub->errno){ throw new Exception('Delete subscription failed: '.$delSub->error); }

            // delete the app user ONLY if it belongs to this promoter
            $delUser = $con->prepare("DELETE FROM `$USER_TABLE` WHERE id=? AND profile_type_id=3 AND profile_id=?");
            $delUser->bind_param("ii",$linked_uid,$id);
            $delUser->execute();
            if ($delUser->errno){ throw new Exception('Delete user failed: '.$delUser->error); }
          }

          // delete promoter profile
          $delProm = $con->prepare("DELETE FROM `$TABLE` WHERE id=?");
          $delProm->bind_param("i",$id);
          $delProm->execute();
          if ($delProm->errno){ throw new Exception('Delete promoter failed: '.$delProm->error); }

          $con->commit();
          redirect_after_action('Deleted');
        } catch (Exception $e){
          $con->rollback();
          $err='Delete failed: '.$e->getMessage();
        }
      }
    }
  }
}

/* -------- UI -------- */
?>
<link rel="stylesheet" href="/adminconsole/assets/ui.css">
<style>
  /* Skill-master-inspired compact layout */
  :root{
    --bg:#0c1722;
    --panel:#0b1220;
    --card:#0f1720;
    --muted:#94a3b8;
    --accent:#10b981;
    --danger:#ef4444;
    --card-radius:10px;       /* slightly smaller radius for skill-master look */
    --inner-shadow: 0 6px 14px rgba(2,6,23,0.5);
  }

  body { background: var(--bg); color: #e6eef8; font-family: "Inter", system-ui, -apple-system, "Segoe UI", Roboto, "Helvetica Neue", Arial; }

  .master-wrap { padding: 20px 24px; max-width:1080px; margin:0 auto; }

  .headbar { display:flex; align-items:center; gap:12px; margin-bottom:14px; }
  .headbar h2 { font-size:18px; margin:0; font-weight:600; color:#e6eef8; }
  .badge { padding:6px 10px; border-radius:8px; font-size:13px; }

  .card {
    background: linear-gradient(180deg, rgba(18,24,33,0.85), rgba(9,13,18,0.85));
    border-radius: var(--card-radius);
    padding: 14px;
    box-shadow: var(--inner-shadow);
    border: 1px solid rgba(255,255,255,0.03);
    margin-bottom:14px;
  }

  .card .card-header {
    display:flex;
    align-items:center;
    justify-content:space-between;
    gap:12px;
    margin-bottom:10px;
  }
  .card .card-header .title {
    font-size:15px;
    font-weight:600;
    color:#e6eef8;
  }
  .card .card-header .subtitle {
    font-size:12px;
    color:var(--muted);
  }

  /* Compact form layout similar to Skill Master */
  .form-row { display:flex; flex-wrap:wrap; gap:10px; align-items:flex-end; }
  .form-row .inp { min-width:220px; display:flex; flex-direction:column; gap:6px; }
  .form-row .inp input[type="text"],
  .form-row .inp textarea,
  .form-row .inp select {
    background: rgba(255,255,255,0.02);
    border: 1px solid rgba(255,255,255,0.04);
    color: #e6eef8;
    padding:8px 10px;
    border-radius:6px;
    font-size:13px;
    outline:none;
    transition: box-shadow .10s, border-color .10s;
    min-height:34px;
  }
  .form-row .inp textarea { min-height:60px; resize:vertical; padding:8px; }
  .form-row .inp input:focus,
  .form-row .inp textarea:focus,
  .form-row .inp select:focus {
    box-shadow: 0 6px 16px rgba(16,185,129,0.07);
    border-color: rgba(16,185,129,0.9);
  }
  .form-row .inp label { font-size:12px; color:var(--muted); display:block; }

  ::placeholder { color: rgba(255,255,255,0.30); }

  /* Buttons - compact */
  .btn {
    display:inline-flex; align-items:center; justify-content:center; gap:8px;
    padding:7px 10px; border-radius:8px; text-decoration:none; border:0;
    font-size:13px; cursor:pointer;
    background: rgba(255,255,255,0.03); color:#dbeafe;
  }
  .btn.success { background: var(--accent); color: #002115; box-shadow: 0 6px 14px rgba(16,185,129,0.10); }
  .btn.secondary { background: rgba(255,255,255,0.04); color:var(--muted); }
  .btn.danger { background: var(--danger); color:#fff; }
  .btn[disabled], button[disabled]{ opacity:.55; cursor:not-allowed; }

  .back-pill { padding:7px 10px; border-radius:8px; background: rgba(255,255,255,0.03); color:var(--muted); border:1px solid rgba(255,255,255,0.02); font-size:13px; }

  /* form actions aligned left and compact (Cancel removed as requested) */
  .form-actions { margin-top:12px; display:flex; gap:10px; align-items:center; }

  /* toolbar / add new */
  .toolbar { display:flex; gap:10px; align-items:center; }

  /* table compact styling to match Skill Master */
  .table { width:100%; border-collapse:collapse; font-size:13px; }
  .table thead tr th { padding:8px 10px; text-align:left; color:var(--muted); font-weight:600; border-bottom:1px solid rgba(255,255,255,0.03); background:var(--panel); position:sticky; top:0; }
  .table tbody td { padding:9px 10px; border-bottom:1px solid rgba(255,255,255,0.02); vertical-align:middle; color:#e6eef8; }

  @media (max-width: 980px){
    .master-wrap { padding:14px; }
    .form-row .inp { min-width:100%; flex:1 1 100%; }
    .card { padding:12px; }
    .table thead tr th { font-size:12px; }
  }

</style>

<div class="master-wrap">
  <div class="headbar">
    <h2 style="margin:0"><?=h($page_title)?></h2>
    <?php if(isset($_GET['ok'])): ?><div class="badge" style="margin-left:12px;background:#14532d;color:#a7f3d0"><?=h($_GET['ok'])?></div><?php endif; ?>
    <?php if($err): ?><div class="badge" style="margin-left:12px;background:#4c0519;color:#fecdd3"><?=h($err)?></div><?php endif; ?>
  </div>

<?php if ($mode_add || $mode_edit):
  // block view if user can't view (but allow add/edit checks handled above)
  if (!$can_view){ echo '<div class="card"><div style="opacity:.8;padding:18px">Permission denied: cannot view this page.</div></div>'; echo '</div>'; echo ob_get_clean(); exit; }

  $row=['id'=>0,'name'=>'','mobile_no'=>'','pan_no'=>'','city_id'=>'','address'=>'','status'=>1];
  if ($mode_edit && $edit_id>0){
    $st=$con->prepare("SELECT id,name,mobile_no,pan_no,city_id,address,status FROM `$TABLE` WHERE id=?");
    $st->bind_param("i",$edit_id); $st->execute();
    $rs=$st->get_result(); if($rs && ($f=$rs->fetch_assoc())) $row=$f;
    // if edit not permitted, inform user (server-side enforcement already done on POST)
    if (!$can_edit){
      $err = $err ?: 'You do not have permission to edit records.';
    }
  }
?>
  <div class="card">
    <div class="card-header">
      <div>
        <div class="title"><?= ($row['id']? 'Edit Promoter' : 'Add Promoter') ?></div>
        <div class="subtitle">Add or update promoter details</div>
      </div>
      <div>
        <?php
          // Back to list: prefer explicit return param if provided, otherwise use script path with add/edit removed
          $back_href = !empty($_GET['return']) ? (string)$_GET['return'] : keep_params(['add'=>null,'edit'=>null]);
        ?>
        <a class="back-pill" href="<?=h($back_href)?>">Back to List</a>
      </div>
    </div>

    <form method="post" action="<?=h(keep_params())?>">
      <input type="hidden" name="csrf_token" value="<?=h(csrf_token())?>">
      <input type="hidden" name="_action" value="save">
      <input type="hidden" name="id" value="<?=h($row['id'])?>">

      <div class="form-row" style="margin-top:6px;">
        <label class="inp">
          <label>Name *</label>
          <input type="text" name="name" value="<?=h($row['name'])?>" required <?= $can_edit || ($row['id']==0 && $can_add) ? '' : 'disabled'?> placeholder="Enter full name">
        </label>

        <label class="inp">
          <label>Mobile *</label>
          <input type="text" name="mobile_no" value="<?=h($row['mobile_no'])?>" required <?= $can_edit || ($row['id']==0 && $can_add) ? '' : 'disabled'?> placeholder="10 digit mobile">
        </label>

        <label class="inp">
          <label>PAN</label>
          <input type="text" name="pan_no" value="<?=h($row['pan_no'])?>" placeholder="ABCDE1234F" <?= $can_edit || ($row['id']==0 && $can_add) ? '' : 'disabled'?>>
        </label>

        <label class="inp">
          <label>City</label>
          <input type="text" name="city" value="<?=h($row['city_id'])?>" placeholder="Keep as string" <?= $can_edit || ($row['id']==0 && $can_add) ? '' : 'disabled'?>>
        </label>

        <label class="inp" style="flex:1 1 100%;">
          <label>Address</label>
          <textarea name="address" rows="2" <?= $can_edit || ($row['id']==0 && $can_add) ? '' : 'disabled'?>><?=h($row['address'])?></textarea>
        </label>

        <label class="inp" style="min-width:160px;">
          <label>Status</label>
          <select name="status" <?= $can_edit || ($row['id']==0 && $can_add) ? '' : 'disabled'?>>
            <option value="1" <?=((int)$row['status']===1?'selected':'')?>>Active</option>
            <option value="0" <?=((int)$row['status']===0?'selected':'')?>>Inactive</option>
          </select>
        </label>
      </div>

      <div class="form-actions" style="margin-top:12px;">
        <?php if (($row['id']==0 && $can_add) || ($row['id']>0 && $can_edit)): ?>
          <button class="btn success" type="submit">Save</button>
        <?php else: ?>
          <button class="btn success" type="button" disabled>Save (disabled)</button>
        <?php endif; ?>

        <!-- Cancel removed as requested. Only Save remains. -->
      </div>
    </form>
  </div>

<?php else:
  if (!$can_view){
    echo '<div class="card"><div style="opacity:.8;padding:18px">Permission denied: cannot view list.</div></div>';
    echo '</div>';
    echo ob_get_clean();
    exit;
  }

  $rows = $con->query("SELECT id,name,mobile_no,pan_no,city_id,orderby,status FROM `$TABLE` ORDER BY orderby ASC,id DESC LIMIT 50");
?>
  <div class="card toolbar">
    <?php if($can_add): ?>
      <a class="btn success" href="<?=h(keep_params(['add'=>1,'edit'=>null]))?>">Add New</a>
    <?php else: ?>
      <button class="btn success" disabled>Add New</button>
    <?php endif; ?>
  </div>
  <div class="card table-wrap">
    <table class="table" style="width:100%; border-collapse:collapse;">
      <thead><tr>
        <th>SR No</th>
        <th>Name</th>
        <th>Mobile</th>
        <th>PAN</th>
        <th>City</th>
        <th>Order</th>
        <th>Status</th>
        <th>Actions</th>
      </tr></thead>
      <tbody>
      <?php $sr=0; while($r=$rows->fetch_assoc()): $sr++; ?>
        <tr>
          <td><?= $sr ?></td>
          <td><?=h($r['name'])?></td>
          <td><?=h($r['mobile_no'])?></td>
          <td><?=h($r['pan_no'])?></td>
          <td><?=h($r['city_id'])?></td>
          <td><?=h($r['orderby'])?></td>
          <td><?=((int)$r['status']===1?'<span class="badge" style="background:#064e3b;color:#a7f3d0;padding:6px 8px;border-radius:8px">Active</span>':'<span class="badge" style="background:#3f1d1d;color:#fecaca;padding:6px 8px;border-radius:8px">Inactive</span>')?></td>
          <td>
            <?php if($can_edit): ?>
              <a class="btn" href="<?=h(keep_params(['edit'=>$r['id'],'add'=>null]))?>">Edit</a>
            <?php else: ?>
              <button class="btn" disabled>Edit</button>
            <?php endif; ?>

            <?php if($can_delete): ?>
            <form method="post" action="<?=h(keep_params())?>" style="display:inline" onsubmit="return confirm('Delete this promoter and all related records?');">
              <input type="hidden" name="csrf_token" value="<?=h(csrf_token())?>">
              <input type="hidden" name="_action" value="delete">
              <input type="hidden" name="id" value="<?=$r['id']?>">
              <button class="btn danger" type="submit">Delete</button>
            </form>
            <?php else: ?>
              <button class="btn danger" disabled>Delete</button>
            <?php endif; ?>
          </td>
        </tr>
      <?php endwhile; if($sr==0): ?><tr><td colspan="8" style="text-align:center;opacity:.7;padding:20px">No Records</td></tr><?php endif; ?>
      </tbody>
    </table>
  </div>
<?php endif; ?>
</div>
<?php echo ob_get_clean(); ?>
